<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://snaatch.com
 * @since      1.0.0
 *
 * @package    Snaatch
 * @subpackage Snaatch/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Snaatch
 * @subpackage Snaatch/includes
 * @author     snaatch GmbH <hello@snaatch.com>
 */
class Snaatch {

  /**
   * The loader that's responsible for maintaining and registering all hooks that power
   * the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      Snaatch_Loader    $loader    Maintains and registers all hooks for the plugin.
   */
  protected $loader;

  /**
   * The unique identifier of this plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $plugin_name    The string used to uniquely identify this plugin.
   */
  protected $plugin_name;

  /**
   * The current version of the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $version    The current version of the plugin.
   */
  protected $version;

  /**
   * Define the core functionality of the plugin.
   *
   * Set the plugin name and the plugin version that can be used throughout the plugin.
   * Load the dependencies, define the locale, and set the hooks for the admin area and
   * the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function __construct() {
    if ( defined( 'SNAATCH_VERSION' ) ) {
      $this->version = SNAATCH_VERSION;
    } else {
      $this->version = '1.0.0';
    }
    $this->plugin_name = 'snaatch';

    $this->load_dependencies();
    $this->set_locale();
    $this->define_admin_hooks();
    $this->define_public_hooks();

  }

  /**
   * Load the required dependencies for this plugin.
   *
   * Include the following files that make up the plugin:
   *
   * - Snaatch_Loader. Orchestrates the hooks of the plugin.
   * - Snaatch_i18n. Defines internationalization functionality.
   * - Snaatch_Admin. Defines all hooks for the admin area.
   * - Snaatch_Public. Defines all hooks for the public side of the site.
   *
   * Create an instance of the loader which will be used to register the hooks
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function load_dependencies() {

    /**
     * The class responsible for orchestrating the actions and filters of the
     * core plugin.
     */
    require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-snaatch-loader.php';

    /**
     * The class responsible for defining internationalization functionality
     * of the plugin.
     */
    require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-snaatch-i18n.php';

    /**
     * The class responsible for defining all actions that occur in the admin area.
     */
    require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-snaatch-admin.php';

    /**
     * The class responsible for defining all actions that occur in the public-facing
     * side of the site.
     */
    require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-snaatch-public.php';

    $this->loader = new Snaatch_Loader();

  }

  /**
   * Define the locale for this plugin for internationalization.
   *
   * Uses the Snaatch_i18n class in order to set the domain and to register the hook
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function set_locale() {

    $plugin_i18n = new Snaatch_i18n();

    $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

  }

  /**
   * Register all of the hooks related to the admin area functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_admin_hooks() {

    $plugin_admin = new Snaatch_Admin( $this->get_plugin_name(), $this->get_version() );

    $this->loader->add_action( 'admin_menu', $plugin_admin, 'register_menu' );
    $this->loader->add_action( 'wp_ajax_import_image', $plugin_admin, 'import_image' );

    add_filter(
      'wp_prepare_attachment_for_js',
      array( self::class, 'add_attachment_json_snaatch_meta' ),
      10,
      3
    );

    /*
    $this->loader->add_filter(
      'wp_get_attachment_image_attributes',
      array( self::class, 'add_snaatch_version_id_class' ),
      10,
      3
    );*/

  }

  /**
   * Adds the class "snaatch_version_id" to attachment images for linked attachments
   *
   * @since 2.0.0
   *
   * @param string[]     $attr
   * @param \WP_Post     $attachment
   * @param string|int[] $size
   * @return string[]
   */
  public static function add_snaatch_version_id_class( $attr, $attachment, $size ) {
    $snaatch_version_id = get_post_meta( $attachment->ID, 'snaatch_version_id', true );
    if ( ! empty( $snaatch_version_id ) ) {
      $attr['class'] .= ' snaatch-file';
    }
    else {
      $attr['class'] .= ' snaatch-file-else';
    }
    return $attr;
  }

  /**
   * Add the snaatch image ID to linked attachments when outputting attachment data via an ajax request.
   *
   * @since 2.0.0
   *
   * @param array    $response
   * @param \WP_Post $attachment
   * @param [type]   $meta
   * @return array
   */
  public static function add_attachment_json_snaatch_meta( $response, $attachment, $meta ) {
    $meta = get_metadata( 'post', $attachment->ID, '', true );
    if ( ! empty( $meta['snaatch_version_id'] ) ) {
      $response['snaatch_version_id'] = $meta['snaatch_version_id'][0];
    }
    return $response;
  }

  /**
   * Register all of the hooks related to the public-facing functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_public_hooks() {

    $plugin_public = new Snaatch_Public( $this->get_plugin_name(), $this->get_version() );

    $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
    $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

  }

  /**
   * Run the loader to execute all of the hooks with WordPress.
   *
   * @since    1.0.0
   */
  public function run() {
    $this->loader->run();
  }

  /**
   * The name of the plugin used to uniquely identify it within the context of
   * WordPress and to define internationalization functionality.
   *
   * @since     1.0.0
   * @return    string    The name of the plugin.
   */
  public function get_plugin_name() {
    return $this->plugin_name;
  }

  /**
   * The reference to the class that orchestrates the hooks with the plugin.
   *
   * @since     1.0.0
   * @return    Snaatch_Loader    Orchestrates the hooks of the plugin.
   */
  public function get_loader() {
    return $this->loader;
  }

  /**
   * Retrieve the version number of the plugin.
   *
   * @since     1.0.0
   * @return    string    The version number of the plugin.
   */
  public function get_version() {
    return $this->version;
  }

}
