<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://snaatch.com
 * @since      1.0.0
 *
 * @package    Snaatch
 * @subpackage Snaatch/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Snaatch
 * @subpackage Snaatch/admin
 * @author     snaatch GmbH <hello@snaatch.com>
 */
class Snaatch_Admin {

  /**
   * The ID of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string $plugin_name The ID of this plugin.
   */
  private $plugin_name;

  /**
   * The version of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string $version The current version of this plugin.
   */
  private $version;

  /**
   * Initialize the class and set its properties.
   *
   * @param string $plugin_name The name of this plugin.
   * @param string $version The version of this plugin.
   * @since    1.0.0
   */
  public function __construct($plugin_name, $version) {

    $this->plugin_name = $plugin_name;
    $this->version = $version;

    add_action('wp_enqueue_media', [&$this, 'enqueue_media']);
    add_action('admin_enqueue_scripts', [&$this, 'enqueue_menu_fix']);
  }

  /**
   * Register the javascript for the admin area to fix menu.
   *
   * @since    1.0.7
   */
  public function enqueue_menu_fix() {
    wp_enqueue_script('snaatch-admin', plugin_dir_url(__FILE__) . 'js/snaatch-menu-fix.js', array('jquery'), $this->version, true);
  }

  /**
   * Register the styles and javascript for Snaatch plugin.
   *
   * @since    1.0.0
   */
  public function enqueue_media() {

    $current_screen = is_admin() && function_exists('get_current_screen') ? get_current_screen()->base : ''; // Current admin screen.
    if ($current_screen !== 'upload') {
      wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/snaatch-admin.css', array(), $this->version, 'all');
      wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/snaatch-admin.js', array('jquery'), $this->version, true);

      // Get the value of 'importAsWebP' setting
      $import_as_webp = get_option('importAsWebP');

      // Get the value of 'useBetaVersion' setting
      $use_beta_version = get_option('useBetaVersion');

      // Pass the setting value to JavaScript using wp_localize_script
      wp_localize_script($this->plugin_name, 'importAsWebP', array(
        'importAsWebP' => $import_as_webp
      ));
      wp_localize_script($this->plugin_name, 'useBetaVersion', array(
        'useBetaVersion' => $use_beta_version
      ));

    }
  }

  public function register_menu() {


    add_menu_page(
      'snaatch | WordPress',     // Page title
      'snaatch',     // Menu title
      'manage_options',  // Capability required to access the menu
      'plugin-menu',     // Menu slug
      [&$this, 'adminHello'], // Callback function to render the menu page
      'https://snaatch.com/favicon-16x16.png'
    );

    $this->register_settings();

    // Add a submenu under the main menu
    add_submenu_page(
      'plugin-menu',      // Parent menu slug
      'Settings',         // Page title
      'Settings',         // Menu title
      'manage_options',   // Capability required to access the menu
      'plugin-settings',  // Menu slug
      [&$this, 'settingsPage']  // Callback function to render the settings page
    );
  }

  public function settingsPage() {
    ?>
    <div class="wrap">
      <h2>Plugin Settings</h2>
      <form method="post" action="options.php">
        <?php
        // Output security fields for the registered settings section
        settings_fields('plugin-settings-group');

        // Output form fields
        do_settings_sections('plugin-settings');

        // Submit button
        submit_button('Save Settings');
        ?>
      </form>
    </div>
    <?php
  }

  public function register_settings() {
    // Register a section for our settings
    add_settings_section(
      'plugin-settings-section',  // ID
      'Plugin Settings',          // Title
      [$this, 'settingsSectionDescription'], // Callback for section description
      'plugin-settings'           // Page slug
    );

    // Register settings fields
    add_settings_field(
      'proxy_url',               // ID
      'Proxy URL',               // Title
      [$this, 'proxyUrlField'],  // Callback for field
      'plugin-settings',         // Page slug
      'plugin-settings-section'  // Section ID
    );

    add_settings_field(
      'ignore_certificate',      // ID
      'Ignore Certificate',      // Title
      [$this, 'ignoreCertificateField'],  // Callback for field
      'plugin-settings',         // Page slug
      'plugin-settings-section'  // Section ID
    );

    add_settings_field(
      'importAsWebP',      // ID
      'Import all Pictures as WebP',      // Title
      [$this, 'importAsWebPField'],  // Callback for field
      'plugin-settings',         // Page slug
      'plugin-settings-section'  // Section ID
    );

    add_settings_field(
      'useBetaVersion',      // ID
      'Use snaatch Beta version',      // Title
      [$this, 'useBetaVersionField'],  // Callback for field
      'plugin-settings',         // Page slug
      'plugin-settings-section'  // Section ID
    );

    // Register the settings with WordPress
    register_setting(
      'plugin-settings-group',   // Option group
      'proxy_url',               // Option name (used in the database)
      ['sanitize_callback' => 'esc_url_raw']  // Sanitization callback
    );

    register_setting(
      'plugin-settings-group',
      'importAsWebP',
      ['sanitize_callback' => 'intval']
    );

    register_setting(
      'plugin-settings-group',
      'useBetaVersion',
      ['sanitize_callback' => 'intval']
    );

    register_setting(
      'plugin-settings-group',
      'ignore_certificate',
      ['sanitize_callback' => 'intval']
    );
  }

  public function settingsSectionDescription() {
    echo 'Setup your Proxy Settings here:<br> <br> <b>Proxy URL</b>: The URL of your proxy server. <br> <b>Ignore Certificate</b>: Ignore the certificate of the proxy server. <br> <br> <b>Example</b>: <br> Proxy URL: http://your-proxy:8080';
  }

  public function proxyUrlField() {
    $proxy_url = esc_url_raw(get_option('proxy_url', ''));

    echo "<input type='text' name='proxy_url' value='$proxy_url' />";
  }

  public function ignoreCertificateField() {
    $ignore_certificate = (get_option('ignore_certificate', 0));

    echo "<input value='1' type='checkbox' name='ignore_certificate' " . checked(1, $ignore_certificate, false) . ' />';
  }

  public function importAsWebPField() {
    $importAsWebP = (get_option('importAsWebP', 0));

    echo "<input value='1' type='checkbox' name='importAsWebP' " . checked(1, $importAsWebP, false) . ' />';
  }

  public function useBetaVersionField() {
    $useBetaVersion = (get_option('useBetaVersion', 0));

    echo "<input value='1' type='checkbox' name='useBetaVersion' " . checked(1, $useBetaVersion, false) . ' />';
  }

  function plugin_name_render_menu() {
    // Code to render the menu page content
  }

  public function adminHello() {
    include 'partials/snaatch-admin-display.php';
  }

  function isBase64Url($url) {
    // Use regex to check if the URL has a data URI scheme with base64
    return preg_match('#^data:[^;]+;base64,#', $url);
  }

  function addQueryParam($url, $paramName, $paramValue) {
    // Parse the URL
    $parsedUrl = parse_url($url);

    // Check if the URL has query parameters
    if (isset($parsedUrl['query'])) {
      // If query parameters exist, append the new parameter
      $parsedUrl['query'] .= "&$paramName=$paramValue";
    } else {
      // If no query parameters exist, create the query string
      $parsedUrl['query'] = "$paramName=$paramValue";
    }

    // Rebuild the URL with the modified query parameters
    $newUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'] . $parsedUrl['path'] . '?' . $parsedUrl['query'];

    // Check if there are fragments (e.g., #section) in the original URL and append them to the new URL
    if (isset($parsedUrl['fragment'])) {
      $newUrl .= '#' . $parsedUrl['fragment'];
    }

    return $newUrl;
  }

  function isFileUrl($url) {
    // Extract the path from the URL
    $path = parse_url($url, PHP_URL_PATH);

    // Get the file extension from the path
    $extension = pathinfo($path, PATHINFO_EXTENSION);

    // Check if the extension is not empty (indicating a file URL)
    return !empty($extension);
  }

  private function get_post_file_value($key, $default = '') {
      return isset($_POST['file'][$key]) ? $_POST['file'][$key] : $default;
  }

  public function import_image() {
    global $wpdb;
    // check for permissions
    if (
      !current_user_can('upload_files')
    ) {
      wp_send_json_error(__('Permission denied', 'snaatch'));
    }

    // check request properties
    if (
      !isset($_POST['file']) ||
      !is_array($_POST['file'])
    ) {
      wp_send_json_error(__('Invalid download request', 'snaatch'));
    }

    // destructure request
    $id = $this->get_post_file_value('id');
    $filename = $this->get_post_file_value('filename');
    $url = $this->get_post_file_value('url');
    $width = $this->get_post_file_value('width');
    $height = $this->get_post_file_value('height');
    $alt = $this->get_post_file_value('alt');
    $title = $this->get_post_file_value('title');
    $description = $this->get_post_file_value('description');
    $filename_with_extension = sanitize_file_name($this->get_post_file_value('filename')) . '.' . $this->get_post_file_value('extension');


    // check required parameters
    if (
      !isset($id) ||
      empty($id) ||
      !isset($filename) ||
      empty($filename) ||
      !isset($url) ||
      empty($url) ||
      !isset($width) ||
      empty($width) ||
      !is_numeric($width) ||
      !isset($height) ||
      empty($height) ||
      !is_numeric($height)
    ) {
      wp_send_json_error(__('Invalid download request. Either id, filename, url, width or height is missing or invalid.', 'snaatch'));
    }

    // use wpdb to query the database
    $query = $wpdb->get_results($wpdb->prepare("SELECT * FROM $wpdb->postmeta WHERE meta_key = 'snaatch_version_id' AND meta_value = %s", $id));
    if(count($query) > 0) {
      // if the image already exists, return as we uploaded it
      wp_send_json_success(array(
        'attachment_id' => $query[0]->post_id,
        'attachment_metadata' => wp_get_attachment_metadata($query[0]->post_id),
        'data' => wp_prepare_attachment_for_js($query[0]->post_id)
      ));
      wp_die();
    }

    // convert $width and $height to integers
    $width = intval($width);
    $height = intval($height);

    // ToDo: check nonce
    $debug_log = array();
    $debug_log[] = $_POST['file'];

    // Get the image's mime type
    $filetype = wp_check_filetype($filename_with_extension, null);
    $debug_log[] = $filetype;

    $debug_log[] = $_POST['file'];

    // declare variables
    $attachment = array();

    // check if URL is base64 url
    $url_is_base64 = $this->isBase64Url($url);
    $debug_log[] = 'url_is_base64: ' . $url_is_base64;

    //TODO: maybe remove base64 handling

    // create if, that is always true
    if ($url_is_base64) {
      $base64_img = $url;

      // Upload dir.
      $upload_dir = wp_upload_dir();
      $upload_path = str_replace('/', DIRECTORY_SEPARATOR, $upload_dir['path']) . DIRECTORY_SEPARATOR;

      // resolve mime type from base64 string
      $mime = substr($base64_img, 11, 4);
      $mime = str_replace(';', '', $mime);
      if ($mime != 'jpeg' && $mime != 'png') {
        return false;
      }
      $img = str_replace('data:image/' . $mime . ';base64,', '', $base64_img);
      $img = str_replace(' ', '+', $img);
      $decoded = base64_decode($img);
      $filename = $title . '.' . $mime;
      $file_type = 'image/' . $mime;
      $clean_filename = sanitize_file_name($filename);
      $unique_filename = wp_unique_filename($upload_dir['path'], $clean_filename);
      $hashed_filename = $unique_filename;

      // 3. Convert File Names
      // Modify the following line where you set $filename_with_extension


      // Save the image in the uploads directory.
      $upload_file = file_put_contents($upload_path . $hashed_filename, $decoded);

      // print the upload path and upload_dir
      $debug_log[] = 'upload_path: ' . $upload_path;

      $attachment = array(
        'post_mime_type' => $file_type,
        'post_title' => preg_replace('/\.[^.]+$/', '', basename($hashed_filename)),
        'post_content' => '',
        'post_status' => 'inherit',
        'guid' => $upload_dir['url'] . '/' . basename($hashed_filename)
      );
    } else {
      $debug_log[] = 'handle normal image';
      $is_sharing_url = $this->isFileUrl($url) ? false : true;
      $debug_log[] = 'is_sharing_url: ' . $is_sharing_url;

      $debug_log[] = 'handle file url';

      // Upload dir.
      $upload_dir = wp_upload_dir();
      $upload_path = str_replace('/', DIRECTORY_SEPARATOR, $upload_dir['path']) . DIRECTORY_SEPARATOR;

      $clean_filename = sanitize_file_name($filename_with_extension);
      $unique_filename = wp_unique_filename($upload_dir['path'], $clean_filename);
      $hashed_filename = $unique_filename;

      // Download file
      $ch = curl_init();

      $proxy = get_option('proxy_url', '');

      // if proxy is set, use it proxy is either '' or 'http://proxy:port'
      if ($proxy != '') {
        curl_setopt($ch, CURLOPT_PROXY, $proxy);
        // if ignore certificate is set, ignore it
        if (get_option('ignore_certificate', 0) == 1) {
          curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
          curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        }
      }

      // Set cURL options
      curl_setopt($ch, CURLOPT_URL, $url);
      curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

      // Execute cURL session
      $file_content = curl_exec($ch);

      // Check for cURL errors
      if (curl_errno($ch)) {
        //JSON encode and send the error success = false
        die(json_encode(array('successs' => false, 'data' => curl_error($ch))));
      }

      // Close cURL session
      curl_close($ch);


      // Save the image in the uploads directory.
      $upload_file = file_put_contents($upload_path . $hashed_filename, $file_content);

      $attachment = array(
        'post_mime_type' => $filetype['type'],
        'post_title' => preg_replace('/\.[^.]+$/', '', basename($hashed_filename)),
        'post_content' => '',
        'post_status' => 'inherit',
        'guid' => $upload_dir['url'] . '/' . basename($hashed_filename)
      );
    }

    // 1. Add the image to the media library
    $attachment_id = wp_insert_attachment($attachment);
    $debug_log[] = 'attachment_id: ' . $attachment_id;

    // 2. Add the same image to the media library
    $success = update_attached_file($attachment_id, $upload_path . $hashed_filename);
    $debug_log[] = 'update_attached_file: ' . $success;
    if ($success === false) {
      wp_send_json_error(array(
        'success' => false,
        'data' => 'Could not update attached file',
        'debug_log' => $debug_log
      ));
    }
    // 3. Add ALT text to the image
    $altMetaId = update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
    $debug_log[] = 'altMetaId: ' . $altMetaId;

    // 4. Add the additional metadata to the image
    $attachment_metadata = array(
      'width' => $width,
      'height' => $height,
      'file' => $filename_with_extension,
    );
    $attachment_metadata['sizes'] = array('full' => $attachment_metadata);
    $debug_log[] = array('attachment_metadata' => $attachment_metadata);
    $debug_log[] = wp_update_attachment_metadata($attachment_id, $attachment_metadata);

    // store additional metadata from snaatch
    update_post_meta($attachment_id, 'snaatch_version_id', $id);

    $debug_log[] = array(
      'post_meta' => get_post_meta($attachment_id),
    );

    // 5. Update the image's title, description, and caption
    $attachment_post = [
      'ID' => $attachment_id,
      'post_title' => $filename, // This will set the title.
      'post_content' => $description, // This will set the description.
      'post_excerpt' => $title // This will set the caption.
    ];
    wp_update_post($attachment_post);

    $debug_log[] = 'attachement_subsizes: ' . $attachment_id;
    $debug_log[] = wp_create_image_subsizes($upload_path . $hashed_filename, $attachment_id);

    wp_send_json_success(array(
      'attachment_id' => $attachment_id,
      'attachment_metadata' => $attachment_metadata,
      'data' => wp_prepare_attachment_for_js($attachment_id),
      'debug_log' => $debug_log
    ));

    wp_die(); // Always use wp_die() at the end of your AJAX function
  }
}
